import numpy as np


def get_tof_data(int_time: float) -> list[np.ndarray]:
    """Generate mock data for a time of flight measurement."""
    nb_pts = int_time * 1e9
    setpoints = np.arange(nb_pts)
    time_of_flight = (
        np.heaviside(setpoints - 200, 0.5) - np.heaviside(setpoints - setpoints.size * 0.7, 0.5)
    ) * 90e-3
    time_of_flight += np.random.normal(loc=0.0, scale=1e-3, size=time_of_flight.size)
    return [time_of_flight, np.zeros_like(time_of_flight)]


def get_resonator_spec_data(
    frequency_setpoints: np.ndarray,
    f_res: float = 220e6,
    alpha: float = 1.14e-6,
    noise_level: float = 0.1,
) -> tuple[np.ndarray, np.ndarray]:
    """Generate mock data for a resonator spectroscopy experiment with configurable resonance."""
    w = 2 * np.pi * frequency_setpoints
    w_r = 2 * np.pi * f_res

    # Correct capacitance and inductance from resonance frequency and alpha
    capacitance = np.sqrt(alpha / (w_r**2))
    inductance = capacitance / alpha  # since alpha = C / L => L = C / alpha

    r_dot = 50e3  # dot resistance
    z_0 = 50  # line impedance

    r_eff = inductance / (capacitance * r_dot)
    z_load = r_eff + 2 * 1j * np.sqrt(inductance / capacitance) * ((w - w_r) / w_r)
    reflectance = (z_load - z_0) / (z_load + z_0)

    # Noise calculation
    noise = (np.max(reflectance) - np.min(reflectance)) * noise_level
    white_noise = np.random.normal(scale=np.abs(noise), size=len(frequency_setpoints))

    return (frequency_setpoints, reflectance + white_noise)


def get_coulomb_peaks_data(
    amp_setpoints: np.ndarray, noise_level: float = 0.1
) -> tuple[np.ndarray, np.ndarray]:
    """Generate mock data for a coulomb peak sweep of the plunger gate of the charge sensing element."""
    e = 1.6e-19  # electron charge
    capacitance = 1e-3  # capacitance
    e_c = e**2 / (2 * capacitance)  # charging energy
    kb_t = 25.7e-3  # product of boltzmann constant and temperature
    g_max = 2.5e-1  # maximum conductance

    sub_arrays = np.split(amp_setpoints * 1000, 5)
    conductance = np.zeros_like(amp_setpoints)
    index = 0
    for sub_array in sub_arrays:
        g_max = g_max + g_max * np.random.uniform(
            0, 4
        )  # introduces some randomization for each simulation
        for amp in sub_array:
            conductance[index] = (
                g_max
                * (e_c / 4 * kb_t)
                * np.cosh(0.01 * (amp - sub_array[round(len(sub_array) / 2)]) / (2 * kb_t)) ** (-2)
            )
            conductance[index] = conductance[index] / (e**2)
            index += 1

    # Noise calculation
    noise = (np.max(conductance) - np.min(conductance)) * noise_level
    white_noise = np.random.normal(scale=np.abs(noise), size=len(conductance))
    return (amp_setpoints, conductance + white_noise)


def get_centered_coulomb_peak_data(
    amp_setpoints: np.ndarray, noise_level: float = 0.1
) -> tuple[np.ndarray, np.ndarray]:
    """Generate mock data for a single coulomb peak for calibration of the readout point."""
    e = 1.6e-19  # electron charge
    capacitance = 1e-3
    e_c = e**2 / (2 * capacitance)  # charging energy
    kb_t = 25.7e-3  # product of boltzmann constant and temperature
    g_max = 25e20  # maximum conductance

    conductance = np.zeros_like(amp_setpoints)
    center_point = round(len(amp_setpoints) / 2) - np.random.uniform(
        -len(amp_setpoints) / 5, len(amp_setpoints) / 5
    )  # introduces some randomization for each simulation
    amp_setpoints = amp_setpoints * 1000
    for index, amp in enumerate(amp_setpoints):
        conductance[index] = (
            g_max
            * (e_c / 4 * kb_t)
            * np.cosh(0.01 * (amp - amp_setpoints[round(center_point)]) / (2 * kb_t)) ** (-2)
        )

    # Noise calculation
    noise = (np.max(conductance) - np.min(conductance)) * noise_level
    white_noise = np.random.normal(scale=np.abs(noise), size=conductance.shape)

    return (amp_setpoints, conductance + white_noise)


def get_charge_stability_diagram_data(
    dot_locations: list, sensor_location: list
) -> np.ndarray | None:
    """Generate mock data for a charge stability diagram of a double quantum dot system."""
    try:
        import pyscipopt  # noqa
        from qdsim import QDDevice, QDSimulator  # noqa
    except ImportError:
        pyscipopt = None

    if pyscipopt:
        qd_system = QDDevice()
        qd_system.set_custom_dot_locations(dot_locations)
        qdsimulator = QDSimulator("Electrons")
        qdsimulator.set_sensor_locations(sensor_location)
        qdsimulator.simulate_charge_stability_diagram(
            qd_device=qd_system,
            v_range_x=[-10, 25],
            v_range_y=[-10, 25],
            n_points_per_axis=100,
            scanning_gate_indexes=[0, 1],
        )
        qdsimulator.plot_charge_stability_diagrams(
            cmapvalue="RdBu_r",
            plot_potential=True,
            gaussian_noise=False,
            white_noise=False,
            pink_noise=False,
        )
        return qdsimulator.occupation_array
    else:
        print("The charge stability diagram will be printed here as a result.")


def get_qubit_spectroscopy_data(
    center_freq: float,
    frequency_setpoints: np.ndarray,
    mw_frequency: float = 5 * np.pi * 2e6,
    artificial_detuning: float = 5 * np.pi * 1e6,
    noise_level: float = 0.1,
) -> tuple[np.ndarray, np.ndarray]:
    """Generate mock data as a result of a single quantum dot qubit spectroscopy experiment."""
    voltage_offset = 50e-3
    del_v = 25e-3  # sensitivity of the voltage signal to impedance change
    qubit_freq = 2 * np.pi * center_freq
    qubit_spectroscopy = voltage_offset - del_v * (
        0.5
        * mw_frequency**2
        / (
            mw_frequency**2
            + (qubit_freq - 2 * np.pi * frequency_setpoints) ** 2
            + artificial_detuning**2
        )
    )

    # Noise calculation
    noise = (np.max(qubit_spectroscopy) - np.min(qubit_spectroscopy)) * noise_level
    white_noise = np.random.normal(scale=np.abs(noise), size=qubit_spectroscopy.shape)

    return (frequency_setpoints, qubit_spectroscopy + white_noise)


def get_1d_rabi_data(
    rabi_method: str,
    pulse_duration: float | np.ndarray,
    pulse_amplitude: float | np.ndarray,
    f_rabi: float = 20e6,  # Rabi frequency for unit amplitude (Hz)
    T2_rabi: float = 40e-6,  # s, decay constant
    noise_level: float = 0.01,
    delta: float = 0.0,  # frequency detuning (Hz) for single-tone Rabi
) -> tuple[np.ndarray, np.ndarray]:
    """
    Generate mock data for 1D Rabi oscillation experiments.

    Supports:
    - 1D Time Rabi
    - 1D Amplitude Rabi

    Parameters
    ----------
    rabi_method: str
        The preferred method for the rabi experiment: "amplitude" or "duration"
    pulse_duration : float or np.ndarray
        Pulse durations (s).
    pulse_amplitude : float or np.ndarray, optional
        Pulse amplitudes (a.u.).
    f_rabi : float
        Base Rabi frequency for amplitude=1 (Hz).
    T2_rabi : float
        Decay constant (s).
    noise_level : float
        Fractional noise level.
    delta : float
        Static detuning for single-tone Rabi (Hz).

    Returns
    -------
    np.ndarray
        Mock Rabi data (1D or 2D).

    """
    # Base Rabi angular frequency (rad/s)
    omega_0 = 2 * np.pi * f_rabi
    if rabi_method == "amplitude":
        if not isinstance(pulse_duration, float):
            raise ValueError(
                "For amplitude rabi method, please enter a single float value for pulse duration."
            )
        omega_r = 2 * np.pi * (f_rabi * pulse_amplitude * 2.5)  # rad/s
        excited_state_prob = np.sin(omega_r * pulse_duration / 2) ** 2
        excited_state_prob *= np.exp(-pulse_duration / T2_rabi)
        noise = (np.max(excited_state_prob) - np.min(excited_state_prob)) * noise_level
        white_noise = np.random.normal(scale=np.abs(noise), size=excited_state_prob.shape)
        return (pulse_amplitude, excited_state_prob + white_noise)
    # 1D Time Rabi
    elif rabi_method == "duration":
        if not isinstance(pulse_amplitude, float) and pulse_amplitude is not None:
            raise ValueError(
                "For amplitude rabi method, please enter a single float value for pulse duration."
            )
        delta = 2 * np.pi * delta  # detuning in rad/s
        omega_eff = np.sqrt(omega_0**2 + delta**2)
        excited_state_prob = (omega_0 / omega_eff) ** 2 * np.sin(
            omega_eff * pulse_duration / 2
        ) ** 2
        excited_state_prob *= np.exp(-pulse_duration / T2_rabi)
        noise = (np.max(excited_state_prob) - np.min(excited_state_prob)) * noise_level
        white_noise = np.random.normal(scale=np.abs(noise), size=pulse_duration.shape)
        return (pulse_duration, excited_state_prob + white_noise)
    else:
        raise ValueError(
            "Please choose the desired 1D Rabi method as a function parameter: rabi_method = amplitude or duration."
        )


def get_2d_rabi_data(
    rabi_method: str,
    pulse_durations: np.ndarray,
    pulse_amplitudes: None | np.ndarray,
    pulse_frequencies: None | np.ndarray,
    f_rabi: float = 20e6,  # Rabi frequency for unit amplitude (Hz)
    T2_rabi: float = 40e-6,  # s, decay constant
    noise_level: float = 0.01,
    delta: float = 0.0,  # frequency detuning (Hz) for single-tone Rabi
) -> np.ndarray | None:
    """
    Generate mock data for 2D Rabi oscillation experiments.

    Supports:
    - 2D Frequency Chevron
    - 2D Amplitude Chevron

    Parameters
    ----------
    rabi_method: str
        The preferred method for the rabi experiment: "frequency" or "amplitude"
    pulse_durations : np.ndarray
        Pulse durations (s).
    pulse_amplitudes : np.ndarray
        Pulse amplitudes (a.u.).
    pulse_frequencies : np.ndarray
        Pulse frequencies (Hz).
    f_rabi : float
        Base Rabi frequency for amplitude=1 (Hz).
    T2_rabi : float
        Decay constant (s).
    noise_level : float
        Fractional noise level.
    delta : float
        Static detuning for single-tone Rabi (Hz).

    Returns
    -------
    np.ndarray
        Mock Rabi data (1D or 2D).

    """
    # Base Rabi angular frequency (rad/s)
    omega_0 = 2 * np.pi * f_rabi
    # 2D Amplitude Chevron
    if rabi_method == "frequency":
        pulse_durations, pulse_frequencies = np.meshgrid(pulse_durations, pulse_frequencies)
        delta = 2 * np.pi * (pulse_frequencies - pulse_frequencies.mean())  # rad/s
        omega_eff = np.sqrt(omega_0**2 + delta**2)
        excited_state_prob = (omega_0 / omega_eff) ** 2 * np.sin(
            omega_eff * pulse_durations / 2
        ) ** 2
        excited_state_prob *= np.exp(-pulse_durations / T2_rabi)
        noise = (np.max(excited_state_prob) - np.min(excited_state_prob)) * noise_level
        white_noise = np.random.normal(scale=np.abs(noise), size=excited_state_prob.shape)
        return excited_state_prob + white_noise
    # 2D Frequency Chevron
    elif rabi_method == "amplitude":
        pulse_durations, pulse_amplitudes = np.meshgrid(pulse_durations, pulse_amplitudes)
        omega_r = 2 * np.pi * (f_rabi * pulse_amplitudes)  # rad/s
        excited_state_prob = np.sin(omega_r * pulse_durations / 2) ** 2
        excited_state_prob *= np.exp(-pulse_durations / T2_rabi)
        noise = (np.max(excited_state_prob) - np.min(excited_state_prob)) * noise_level
        white_noise = np.random.normal(scale=np.abs(noise), size=excited_state_prob.shape)
        return excited_state_prob + white_noise
    else:
        raise ValueError(
            "Please choose the desired 2D Rabi method as a function parameter: rabi_method = frequency or amplitude."
        )


def get_t1_data(
    wait_setpoints: np.ndarray,
    T1: float = 5e-6,
    noise_level: float = 0.02,
) -> tuple[np.ndarray, np.ndarray]:
    """
    Generate simulated data for a T1 relaxation experiment.

    Parameters
    ----------
    wait_setpoints : np.ndarray
        Array of wait times after excitation.
    T1 :
        Energy relaxation time of the qubit [s].
    noise_level : float
        Relative noise level (fraction of signal amplitude).

    Returns
    -------
        Wait times and simulated measurement values with noise.

    """
    # Ideal T1 decay: excited state probability
    t1_result = np.exp(-wait_setpoints / T1)

    # Add Gaussian white noise scaled to signal amplitude
    noise = (np.max(t1_result) - np.min(t1_result)) * noise_level
    white_noise = np.random.normal(scale=np.abs(noise), size=t1_result.shape)

    return (wait_setpoints, t1_result + white_noise)


def get_ramsey_spec_data(
    wait_setpoints: np.ndarray,
    detuning: float = 10e6,
    T2_star: float = 1e-6,
    noise_level: float = 0.1,
) -> tuple[np.ndarray, np.ndarray]:
    """Generate simulated data for a Ramsey experiment."""
    ramsey_result = 0.5 * (
        1 + np.exp(-wait_setpoints / T2_star) * np.cos(detuning * wait_setpoints)
    )

    # Noise calculation
    noise = (np.max(ramsey_result) - np.min(ramsey_result)) * noise_level
    white_noise = np.random.normal(scale=np.abs(noise), size=ramsey_result.shape)

    return (wait_setpoints, ramsey_result + white_noise)


def get_ramsey_chevron_data(
    wait_setpoints: np.ndarray,
    pulse_freq_setpoints: np.ndarray,
    f_res: float = 5.25e9,
    T2_star: float = 1e-6,
    noise_level: float = 0.01,
) -> np.ndarray:
    """Generate mock data as a result of a Ramsey Chevron experiment."""
    pulse_freq_setpoints, wait_setpoints = np.meshgrid(pulse_freq_setpoints, wait_setpoints)
    ramsey_chevron = 0.5 * (
        1
        + np.exp(-wait_setpoints / T2_star)
        * np.cos((f_res - pulse_freq_setpoints) * wait_setpoints)
    )

    # Noise calculation
    noise = (np.max(ramsey_chevron) - np.min(ramsey_chevron)) * noise_level
    white_noise = np.random.normal(scale=np.abs(noise), size=ramsey_chevron.shape)

    return ramsey_chevron + white_noise


def get_hahn_echo_data(
    wait_setpoints: np.ndarray, T2_echo: float = 1e-5, noise_level: float = 0.1
) -> tuple[np.ndarray, np.ndarray]:
    """Generate mock data as a result of a Hahn Echo experiment."""
    hahn_echo_response = 0.5 * (1 + np.exp(-wait_setpoints / T2_echo))

    # Noise calculation
    noise = (np.max(hahn_echo_response) - np.min(hahn_echo_response)) * noise_level
    white_noise = np.random.normal(scale=np.abs(noise), size=hahn_echo_response.shape)

    return (wait_setpoints, hahn_echo_response + white_noise)


def get_cpmg_data(
    n_gates: int, wait_setpoints: np.ndarray, T2_CPMG: float = 1e-4, noise_level: float = 0.1
) -> tuple[np.ndarray, np.ndarray]:
    """Generate mock data as a result of a Hahn Echo experiment."""
    cpmg_response = 0.5 * (1 + np.exp(-n_gates * wait_setpoints / T2_CPMG))

    # Noise calculation
    noise = (np.max(cpmg_response) - np.min(cpmg_response)) * noise_level
    white_noise = np.random.normal(scale=np.abs(noise), size=cpmg_response.shape)

    return (wait_setpoints, cpmg_response + white_noise)
